<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
include('include/config.php');  // $con = new mysqli(...)

// ---------------------------------------------
// 1) Fetch all medicines for the dropdown
// ---------------------------------------------
$medicines = [];
$medResult = $con->query("SELECT id, name FROM medicine ORDER BY name ASC");
while ($m = $medResult->fetch_assoc()) {
    $medicines[] = $m;
}

// ---------------------------------------------
// 2) Fetch all departments for the dropdown
// ---------------------------------------------
$deptList = [];
$deptResult = $con->query("SELECT id, dname FROM departments ORDER BY dname ASC");
while ($d = $deptResult->fetch_assoc()) {
    $deptList[] = $d;
}

// ---------------------------------------------
// 3) Handle form submission
// ---------------------------------------------
$alertHtml = "";
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['transfer_submit'])) {
    try {
        // a) Gather & sanitize inputs
        $medicine_id = (int) ($_POST['medicine_id'] ?? 0);
        $from_department = (int) ($_POST['from_department'] ?? 0);
        $to_department = (int) ($_POST['to_department'] ?? 0);
        $transfer_qty = (int) ($_POST['quantity'] ?? 0);

        // b) Basic validations
        if ($medicine_id <= 0 || $from_department <= 0 || $to_department <= 0 || $transfer_qty <= 0) {
            throw new Exception("All fields are required and quantity must be a positive number.");
        } 
        if ($from_department === $to_department) {
            throw new Exception("Source and destination departments cannot be the same.");
        }

        // c) Check source department's current stock
        $stmt = $con->prepare("
            SELECT stock_quantity 
            FROM department_medicine_stock 
            WHERE department_id = ? AND medicine_id = ?
        ");
        $stmt->bind_param('ii', $from_department, $medicine_id);
        $stmt->execute();
        $stmt->store_result();

        if ($stmt->num_rows === 0) {
            throw new Exception("Source department has no stock record for this medicine.");
        }

        $stmt->bind_result($currentStock);
        $stmt->fetch();
        $stmt->close();

        if ($currentStock < $transfer_qty) {
            throw new Exception("Not enough stock in source department. (Available: $currentStock, Requested: $transfer_qty)");
        }

        // d) Begin transaction
        $con->begin_transaction();

        // d.1) Decrement source department's stock
        $upd1 = $con->prepare("
            UPDATE department_medicine_stock
            SET stock_quantity = stock_quantity - ?
            WHERE department_id = ? AND medicine_id = ?
        ");
        $upd1->bind_param('iii', $transfer_qty, $from_department, $medicine_id);
        $ok1 = $upd1->execute();
        $upd1->close();

        // d.2) Increment destination department's stock (or insert if missing)
        $stmt2 = $con->prepare("
            SELECT id 
            FROM department_medicine_stock 
            WHERE department_id = ? AND medicine_id = ?
        ");
        $stmt2->bind_param('ii', $to_department, $medicine_id);
        $stmt2->execute();
        $stmt2->store_result();

        if ($stmt2->num_rows === 0) {
            // Insert new record
            $stmt2->close();
            $ins = $con->prepare("
                INSERT INTO department_medicine_stock 
                (department_id, medicine_id, stock_quantity) 
                VALUES (?, ?, ?)
            ");
            $ins->bind_param('iii', $to_department, $medicine_id, $transfer_qty);
            $ok2 = $ins->execute();
            $ins->close();
        } else {
            // Update existing record
            $stmt2->bind_result($dummy);
            $stmt2->fetch();
            $stmt2->close();
            $upd2 = $con->prepare("
                UPDATE department_medicine_stock
                SET stock_quantity = stock_quantity + ?
                WHERE department_id = ? AND medicine_id = ?
            ");
            $upd2->bind_param('iii', $transfer_qty, $to_department, $medicine_id);
            $ok2 = $upd2->execute();
            $upd2->close();
        }

        // d.3) Insert into stock_transfer_log
        $logStmt = $con->prepare("
            INSERT INTO stock_transfer_log 
            (medicine_id, from_department_id, to_department_id, quantity) 
            VALUES (?, ?, ?, ?)
        ");
        $logStmt->bind_param("iiii", $medicine_id, $from_department, $to_department, $transfer_qty);
        $ok3 = $logStmt->execute();
        $logStmt->close();

        // d.4) Commit or rollback
        if ($ok1 && $ok2 && $ok3) {
            $con->commit();
            $alertHtml = "
            <div class='alert alert-success'>
                ✅ Stock transfer successful and logged.
            </div>";
        } else {
            $con->rollback();
            throw new Exception("Transfer failed. Changes rolled back.");
        }

    } catch (Exception $e) {
        if (isset($con) && $con->errno) {
            $alertHtml = "
            <div class='alert alert-danger'>
                ❌ Database error: " . htmlspecialchars($con->error) . "
            </div>";
        } else {
            $alertHtml = "
            <div class='alert alert-danger'>
                ❌ " . htmlspecialchars($e->getMessage()) . "
            </div>";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Stock Transfer</title>
    <link href="http://fonts.googleapis.com/css?family=Lato:300,400,400italic,600,700|Raleway:300,400,500,600,700|Crete+Round:400italic" rel="stylesheet" type="text/css" />
    <link rel="stylesheet" href="vendor/bootstrap/css/bootstrap.min.css">
    <link rel="stylesheet" href="vendor/fontawesome/css/font-awesome.min.css">
    <link rel="stylesheet" href="vendor/themify-icons/themify-icons.min.css">
    <link href="vendor/animate.css/animate.min.css" rel="stylesheet" media="screen">
    <link href="vendor/perfect-scrollbar/perfect-scrollbar.min.css" rel="stylesheet" media="screen">
    <link href="vendor/switchery/switchery.min.css" rel="stylesheet" media="screen">
    <link href="vendor/select2/select2.min.css" rel="stylesheet" media="screen">
    <link href="vendor/bootstrap-datepicker/bootstrap-datepicker3.standalone.min.css" rel="stylesheet" media="screen">
    <link rel="stylesheet" href="assets/css/styles.css">
    <link rel="stylesheet" href="assets/css/plugins.css">
    <link rel="stylesheet" href="assets/css/themes/theme-1.css" id="skin_color" />
    <style>
        .btn-primary {
            margin-left: 980px;
        }
    </style>
</head>
<body>
<div id="app">
    <?php include('include/sidebar.php'); ?>
    <div class="app-content">
        <?php include('include/header.php'); ?>
        <div class="main-content">
            <div class="wrap-content container" id="container">

                <!-- PAGE TITLE -->
                <section id="page-title">
                    <div class="row">
                        <div class="col-sm-8">
                            <h1 class="mainTitle">Stock Transfer</h1>
                        </div>
                    </div>
                </section>

                <div class="container-fluid container-fullw bg-white">

                    <!-- Display Success/Error Alert -->
                    <?= $alertHtml ?>

                    <!-- Stock Transfer Form -->
                    <div class="card mb-4">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0"><strong>Transfer Stock Between Departments</strong></h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="transfer-stock.php">
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label for="medicine_id" class="form-label">Select Medicine</label>
                                        <select class="form-control" id="medicine_id" name="medicine_id" required>
                                            <option value="">-- Choose Medicine --</option>
                                            <?php foreach ($medicines as $m): ?>
                                                <option value="<?= htmlspecialchars($m['id']) ?>">
                                                    <?= htmlspecialchars($m['name']) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-4">
                                        <label for="from_department" class="form-label">From Department</label>
                                        <select class="form-control" id="from_department" name="from_department" required>
                                            <option value="">-- Choose Source Dept --</option>
                                            <?php foreach ($deptList as $d): ?>
                                                <option value="<?= htmlspecialchars($d['id']) ?>">
                                                    <?= htmlspecialchars($d['dname']) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-4">
                                        <label for="to_department" class="form-label">To Department</label>
                                        <select class="form-control" id="to_department" name="to_department" required>
                                            <option value="">-- Choose Destination Dept --</option>
                                            <?php foreach ($deptList as $d): ?>
                                                <option value="<?= htmlspecialchars($d['id']) ?>">
                                                    <?= htmlspecialchars($d['dname']) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label for="quantity" class="form-label">Quantity to Transfer</label>
                                        <input type="number" class="form-control" id="quantity" name="quantity" min="1" required>
                                    </div>
                                </div>
                                <br>
                                <button type="submit" name="transfer_submit" class="btn btn-success">
                                    <i class="fa fa-exchange me-1"></i> Transfer Stock
                                </button>
                            </form>
                        </div>
                    </div>
                    <br>
                    <!-- Recent Stock Transfer Log -->
                    <div class="text-end mb-3">
                        <button onclick="printSection('print-area')" class="btn btn-primary">
                            <i class="ti-printer"></i> Print
                        </button>
                    </div>
                    <div class="card mb-4">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="mb-0"><strong>Recent Stock Transfers</strong></h5>
                        </div>
                        <div class="card-body">
                            <div id="print-area">
                                <table class="table table-bordered table-striped">
                                    <thead class="table-light">
                                        <tr>
                                            <th>No</th>
                                            <th>Medicine</th>
                                            <th>From Dept</th>
                                            <th>To Dept</th>
                                            <th>Quantity</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php
                                        $logResult = $con->query("
                                            SELECT 
                                                l.id,
                                                m.name AS med_name,
                                                d1.dname AS from_dept,
                                                d2.dname AS to_dept,
                                                l.quantity,
                                                l.transfer_date
                                            FROM stock_transfer_log l
                                            JOIN medicine m ON m.id = l.medicine_id
                                            JOIN departments d1 ON d1.id = l.from_department_id
                                            JOIN departments d2 ON d2.id = l.to_department_id
                                            ORDER BY l.transfer_date DESC
                                            LIMIT 10
                                        ");

                                        if ($logResult && $logResult->num_rows === 0) {
                                            echo "
                                            <tr>
                                                <td colspan='6' class='text-center py-3'>No stock transfers found.</td>
                                            </tr>";
                                        } else {
                                            $j = 1;
                                            while ($rowLog = $logResult->fetch_assoc()) {
                                                echo "
                                                <tr>
                                                    <td>{$j}</td>
                                                    <td>" . htmlspecialchars($rowLog['med_name']) . "</td>
                                                    <td>" . htmlspecialchars($rowLog['from_dept']) . "</td>
                                                    <td>" . htmlspecialchars($rowLog['to_dept']) . "</td>
                                                    <td>" . (int)$rowLog['quantity'] . "</td>
                                                    <td>" . htmlspecialchars($rowLog['transfer_date']) . "</td>
                                                </tr>";
                                                $j++;
                                            }
                                        }
                                        ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                </div> <!-- /.container-fullw -->
            </div> <!-- /.wrap-content -->
        </div> <!-- /.main-content -->

        <?php include('include/footer.php'); ?>
        <?php include('include/setting.php'); ?>
    </div> <!-- /.app-content -->
</div> <!-- /#app -->

<!-- SCRIPTS -->
<script src="vendor/jquery/jquery.min.js"></script>
<script src="vendor/bootstrap/js/bootstrap.min.js"></script>
<script src="vendor/modernizr/modernizr.js"></script>
<script src="vendor/jquery-cookie/jquery.cookie.js"></script>
<script src="vendor/perfect-scrollbar/perfect-scrollbar.min.js"></script>
<script src="vendor/switchery/switchery.min.js"></script>
<script src="vendor/maskedinput/jquery.maskedinput.min.js"></script>
<script src="vendor/bootstrap-touchspin/jquery.bootstrap-touchspin.min.js"></script>
<script src="vendor/autosize/autosize.min.js"></script>
<script src="vendor/selectFx/classie.js"></script>
<script src="vendor/selectFx/selectFx.js"></script>
<script src="vendor/select2/select2.min.js"></script>
<script src="vendor/bootstrap-datepicker/bootstrap-datepicker.min.js"></script>
<script src="vendor/bootstrap-timepicker/bootstrap-timepicker.min.js"></script>
<script src="assets/js/main.js"></script>
<script src="assets/js/form-elements.js"></script>

<script>
    jQuery(document).ready(function () {
        Main.init();
        FormElements.init();
    });

    function printSection(divId) {
        var printContents = document.getElementById(divId).innerHTML;
        var win = window.open('', '', 'height=700,width=900');
        win.document.write('<html><head><title>Stock Transfer History</title>');
        win.document.write('<link rel="stylesheet" href="vendor/bootstrap/css/bootstrap.min.css">');
        win.document.write('<style>table { width: 100%; border-collapse: collapse; } table, th, td { border: 1px solid black; padding: 8px; }</style>');
        win.document.write('</head><body>');
        win.document.write('<h3 class="text-center">Stock Transfer History</h3>');
        win.document.write(printContents);
        win.document.write('</body></html>');
        win.document.close();
        win.focus();
        win.print();
        win.close();
    }
</script>
</body>
</html>